<?php

namespace App\Models;

use EloquentFilter\Filterable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Product extends Model
{
    use Filterable;

    protected $fillable = [
        'name',
        'description',
        'slug',
        'category_id',
        'supplier_id', 'warehouse_id',
        'low_qty_alert',
        'image',
        'status',
        'has_variants',
        'sku',
        'upc',
        'tax_type',
        'serial_number',
        'identity',
        'barcode_symbology',
        'price',
        'cost',
        'is_serial_based',
        'warranty_label',
        'warranty_value',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'status' => 'boolean',
        'has_variants' => 'boolean',
        'tax_type' => 'boolean',
        'is_serial_based' => 'boolean',
    ];

    /**
     * Product stocks
     *
     * @return     HasMany  The has many.
     */
    public function stocks(): HasMany
    {
        return $this->hasMany(Stock::class);
    }

    /**
     * Product attributes
     *
     * @return     HasMany  The has many.
     */
    public function productAttributes(): HasMany
    {
        return $this->hasMany(ProductAttribute::class);
    }

    /**
     * Product category
     *
     * @return     BelongsTo  The belongs to.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Product supplier
     *
     * @return     BelongsTo  The belongs to.
     */
    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * User avatar url
     *
     * @return string
     */
    public function getImageUrl(): string
    {
        return $this->image
        ? url('/storage/' . $this->image)
        : asset('images/default/product.png');
    }

    public function getTotalInStock()
    {
        return $this->stocks->sum('quantity');
    }

    public function getTotalSoldStock()
    {
        return $this->stocks->sum('sold');
    }

    public function previous()
    {
        $item = $this->where('id', '<', $this->id)->orderBy('id', 'desc')->first();
        return $item ? $item->id : null;
    }

    public function next()
    {
        $item = $this->where('id', '>', $this->id)->orderBy('id', 'asc')->first();
        return $item ? $item->id : null;
    }
}
