<?php

namespace App\Http\Controllers;

use App\Models\CashBook;
use App\Models\Setting;
use App\Models\User;
use App\Models\UserRole;
use dacoto\EnvSet\Facades\EnvSet;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ApiController extends Controller
{
    /**
     * Provides default settings for all controllers
     * extended by controller
     *
     * @return object
     */
    protected function master(): object
    {
        return Setting::find(1);
    }

    /**
     * System configuration
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return   mixed
     */
    public function protection(Request $request)
    {
        EnvSet::setKey(strtoupper('app_pack'), $request->package_hash ?? null);
        EnvSet::setKey(strtoupper('app_debug'), $request->app_debug ?? false);
        EnvSet::save();
        return redirect('/admin?verison=' . config('app.version') . '&message=Application_configuration_saved_successfully');
    }

    /**
     * DataTable sorting for common resources
     *
     * @param mixed $request
     *
     * @return array
     */
    protected function sort($request): array
    {
        return $request->get('sort', json_decode(json_encode(['order' => 'asc', 'column' => 'created_at']),
            true,
            512,
            JSON_THROW_ON_ERROR
        ));
    }

    /**
     * Generate pagination for common dataTables
     * @param \Illuminate\Database\Eloquent\Collection $items
     *
     * @return array
     */
    protected function pagination($items): array
    {
        return [
            'currentPage' => $items->currentPage(),
            'perPage' => $items->perPage(),
            'total' => $items->total(),
            'totalPages' => $items->lastPage(),
        ];
    }

    protected function channelConfigs($user)
    {
        return $this->master()->getNotificationConfig($user);
    }

    protected function getUsers($admin = false)
    {
        return User::whereIn('role_id', UserRole::where('admin_access', $admin)->pluck('id'))->where('status', true)->get();
    }

    protected function activeRegister()
    {
        return CashBook::where('closed', false)->first();
    }

    protected function getCurrentUserCashbook()
    {
        return $this->activeRegister() ? $this->activeRegister()->uuid : null;
    }

    protected function getActiveRegisterIdentity()
    {
        return $this->activeRegister() ? $this->activeRegister()->id : 0;
    }

    protected function getUniqueBarcode()
    {
        return Str::limit(time() + random_int(100000000000000, 999999999999999), config('product.barcode.length', 12), false);
    }

    protected function calculateAmountFromModel($modelClass, $key, $duration = 'year', $isLast = false)
    {
        return $modelClass::duration($duration, $isLast)->sum($key);
    }

    protected function generateChartFromModel($modelClass, $key = "total_payable_amount", $duration = 'year', $isLast = false)
    {
        $totalAmounts = $modelClass::duration($duration, $isLast)
            ->selectRaw("SUM($key) as $key, MONTH(created_at) as month")
            ->groupBy('month')
            ->pluck($key, 'month')
            ->toArray();

        $graph = [];
        for ($month = 1; $month <= 12; $month++) {
            $graph[] = $totalAmounts[$month] ?? 0;
        }
        return $graph;
    }
}
