<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\WarehouseStoreRequest;
use App\Http\Requests\WarehouseUpdateRequest;
use App\Http\Resources\WarehouseResource;
use App\Models\Warehouse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class WarehouseController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $warehouses = Warehouse::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => WarehouseResource::collection($warehouses->items()),
                'pagination' => $this->pagination($warehouses),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\WarehouseStoreRequest  $request  The request
     *
     * @return     JsonResponse                              The json response.
     */
    public function store(WarehouseStoreRequest $request): JsonResponse
    {
        $warehouse = Warehouse::create($request->validated());
        return response()->json([
            'message' => __('Data saved successfully'),
            'warehouse' => new WarehouseResource($warehouse),
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\Warehouse  $warehouse  The warehouse
     *
     * @return     JsonResponse           The json response.
     */
    public function show(Warehouse $warehouse): JsonResponse
    {
        return response()->json(new WarehouseResource($warehouse));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\WarehouseUpdateRequest  $request    The request
     * @param      \App\Models\Warehouse                      $warehouse  The warehouse
     *
     * @return     JsonResponse                               The json response.
     */
    public function update(WarehouseUpdateRequest $request, Warehouse $warehouse): JsonResponse
    {
        $warehouse->update($request->validated());
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Destroys the given warehouse.
     *
     * @param      \App\Models\Warehouse  $warehouse  The warehouse
     *
     * @return     JsonResponse           The json response.
     */
    public function destroy(Warehouse $warehouse): JsonResponse
    {

        if ($warehouse->products->count() < 1 && $warehouse->productAttributes->count() < 1 && $warehouse->stocks->count() < 1) {
            $warehouse->delete();
            return response()->json([
                'message' => __('Data removed successfully'),
            ]);
        }
        return response()->json([
            'message' => __('Unable to remove it is being used'),
        ], 403);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        $warehouses = Warehouse::whereIn('id', $request->rows)->get();
        foreach ($warehouses as $key => $warehouse) {
            if ($warehouse->products->count() < 1 && $warehouse->productAttributes->count() < 1 && $warehouse->stocks->count() < 1) {
                $warehouse->delete();
            }
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    /**
     * Warehouses list
     *
     * @return     JsonResponse  The json response.
     */
    public function warehouseList(): JsonResponse
    {
        return response()->json(WarehouseResource::collection(Warehouse::get()));
    }
}
