<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\UserRoleStoreRequest;
use App\Http\Requests\UserRoleUpdateRequest;
use App\Http\Resources\UserRoleEditResource;
use App\Http\Resources\UserRoleResource;
use App\Models\User;
use App\Models\UserRole;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class UserRoleController extends ApiController
{
    /**
     * Construct middleware
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware(
            'demo',
            ['only' => ['update', 'destroy']]
        );
    }

    /**
     * User roles list for management
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $roles = UserRole::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));

        return response()->json(
            [
                'items' => UserRoleResource::collection($roles->items()),
                'pagination' => $this->pagination($roles),
            ]
        );
    }

    /**
     * Store role to database
     *
     * @param      \App\Http\Requests\UserRoleStoreRequest  $request  The request
     *
     * @return     JsonResponse                             The json response.
     */
    public function store(UserRoleStoreRequest $request): JsonResponse
    {
        $userRole = UserRole::create($request->validated());
        return response()->json(
            [
                'message' => __('Data saved successfully'),
                'userRole' => new UserRoleEditResource($userRole),
            ]
        );
    }

    /**
     * Display specific role
     *
     * @param      \App\Models\UserRole  $userRole  The user role
     *
     * @return     JsonResponse          The json response.
     */
    public function show(UserRole $userRole): JsonResponse
    {
        if ($userRole->is_primary) {
            return response()->json(
                ['message' => __('Cannot edit a system base function')],
                406
            );
        }
        return response()->json(new UserRoleEditResource($userRole));
    }

    /**
     * Update user role
     *
     * @param      \App\Http\Requests\UserRoleUpdateRequest  $request   The request
     * @param      \App\Models\UserRole                      $userRole  The user role
     *
     * @return     JsonResponse                              The json response.
     */
    public function update(UserRoleUpdateRequest $request, UserRole $userRole): JsonResponse
    {
        $userRole->update($request->validated());
        return response()->json(
            [
                'message' => __('Data updated successfully'),
            ]
        );
    }

    /**
     * Destroys the given user role.
     *
     * @param      \App\Models\UserRole  $userRole  The user role
     *
     * @return     JsonResponse          The json response.
     */
    public function destroy(UserRole $userRole): JsonResponse
    {
        if ($userRole->is_primary || ((int) $this->master()->app_default_role === $userRole->id)) {
            return response()->json(
                ['message' => __('Can not delete a default role')],
                406
            );
        }
        User::where('role_id', $userRole->id)->update(
            ['role_id' => $this->master()->app_default_role]
        );
        $userRole->delete();
        return response()->json(
            ['message' => __('Data removed successfully')]
        );
    }

    /**
     * Gives permissions keys
     *
     * @return     JsonResponse  The json response.
     */
    public function permissions(): JsonResponse
    {
        $permissions = [
            ['key' => 'dashboard', 'label' => __('Dashboard')],

            ['key' => 'sales', 'label' => __('Sales invoices')],
            ['key' => 'remove_sales', 'label' => __('Remove sale invoices ')],

            ['key' => 'repairs', 'label' => __('Repair invoices')],
            ['key' => 'remove_repairs', 'label' => __('Remove repair order ')],

            ['key' => 'bills', 'label' => __('Custom bill invoices')],
            ['key' => 'remove_custom_bills', 'label' => __('Remove custom bills ')],

            ['key' => 'suppliers', 'label' => __('Suppliers')],
            ['key' => 'customers', 'label' => __('Customers')],
            ['key' => 'users', 'label' => __('Users')],
            ['key' => 'roles', 'label' => __('User roles')],

            ['key' => 'products', 'label' => __('Manage products')],
            ['key' => 'categories', 'label' => __('Manage categories')],
            ['key' => 'warehouses', 'label' => __('Manage warehouses')],

            ['key' => 'label_generation', 'label' => __('Label generation')],

            ['key' => 'overall_report', 'label' => __('Overall report')],

            ['key' => 'stock_alerts_report', 'label' => __('Stock alerts')],

            ['key' => 'app_setup', 'label' => __('App setup')],
            ['key' => 'app_configs', 'label' => __('App configuration')],
            ['key' => 'printing_setup', 'label' => __('Print setup')],
            ['key' => 'terms', 'label' => __('Invoice terms')],

            ['key' => 'database_backup', 'label' => __('Database backup')],
            ['key' => 'translations', 'label' => __('Translations')],
            ['key' => 'terminal', 'label' => __('System terminal')],

            ['key' => 'pos_access', 'label' => __('POS access')],

            ['key' => 'discount_implementation', 'label' => __('Discount implementation')],
            ['key' => 'tax_implementation', 'label' => __('Tax implementation')],

        ];

        return response()->json($permissions);
    }
}
