<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\SupplierStoreRequest;
use App\Http\Requests\SupplierUpdateRequest;
use App\Http\Resources\SupplierDetailResource;
use App\Http\Resources\SupplierResource;
use App\Models\Supplier;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SupplierController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $suppliers = Supplier::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => SupplierResource::collection($suppliers->items()),
                'pagination' => $this->pagination($suppliers),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\SupplierStoreRequest  $request  The request
     *
     * @return     JsonResponse                             The json response.
     */
    public function store(SupplierStoreRequest $request): JsonResponse
    {
        $supplier = Supplier::create($request->validated());
        return response()->json([
            'message' => __('Data saved successfully'),
            'supplier' => new SupplierDetailResource($supplier),
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\Supplier  $supplier  The supplier
     *
     * @return     JsonResponse          The json response.
     */
    public function show(Supplier $supplier): JsonResponse
    {
        return response()->json(new SupplierDetailResource($supplier));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\SupplierUpdateRequest  $request   The request
     * @param      \App\Models\Supplier                      $supplier  The supplier
     *
     * @return     JsonResponse                              The json response.
     */
    public function update(SupplierUpdateRequest $request, Supplier $supplier): JsonResponse
    {
        $supplier->update($request->validated());
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Destroys the given supplier.
     *
     * @param      \App\Models\Supplier  $supplier  The supplier
     *
     * @return     JsonResponse          The json response.
     */
    public function destroy(Supplier $supplier): JsonResponse
    {
        if ($supplier->products->count() < 1 && $supplier->productAttributes->count() < 1) {
            $supplier->delete();
            return response()->json([
                'message' => __('Data removed successfully'),
            ]);
        }
        return response()->json([
            'message' => __('Unable to remove it is being used'),
        ], 403);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        $suppliers = Supplier::whereIn('id', $request->rows)->get();
        foreach ($suppliers as $key => $supplier) {
            if ($supplier->products->count() < 1 && $supplier->productAttributes->count() < 1) {
                $supplier->delete();
            } else {
                return response()->json([
                    'message' => __('Unable to remove it is being used'),
                ], 403);
            }
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    public function suppliers(): JsonResponse
    {
        return response()->json(SupplierResource::collection(Supplier::get()));
    }
}
