<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\StockRequest;
use App\Http\Resources\ProductSelectResource;
use App\Http\Resources\ProductStockResource;
use App\Http\Resources\StockDetailResource;
use App\Http\Resources\StockResource;
use App\Http\Resources\WarehouseResource;
use App\Models\Product;
use App\Models\Stock;
use App\Models\Warehouse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class StockController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return     JsonResponse  The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $stocks = Stock::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => StockResource::collection($stocks->items()),
                'pagination' => $this->pagination($stocks),
            ]
        );
    }

    /**
     * Store a newly created resource in storage
     *
     * @param      \App\Http\Requests\StockRequest  $request  The request
     *
     * @return     JsonResponse                          The json response.
     */
    public function store(StockRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $validated['tracking'] = time();
        $validated['uuid'] = \Str::orderedUuid();
        $stock = Stock::create($validated);
        return response()->json([
            'message' => 'Data saved successfully',
            'stock' => new ProductStockResource($stock),
        ]);
    }

    /**
     *  Display the specified resource.
     *
     * @param      \App\Models\Stock  $stock  The stock
     *
     * @return     JsonResponse       The json response.
     */
    public function show(Stock $stock): JsonResponse
    {
        return response()->json(new StockDetailResource($stock));
    }

    /**
     *Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\StockRequest  $request  The request
     * @param      \App\Models\Stock                      $stock    The stock
     *
     * @return     JsonResponse                           The json response.
     */
    public function update(StockRequest $request, Stock $stock): JsonResponse
    {
        $validated = $request->validated();
        $stock->update($validated);
        return response()->json([
            'message' => 'Data updated successfully',
        ]);
    }

    /**
     * Destroys the given stock.
     *
     * @param      \App\Models\Stock  $stock  The stock
     *
     * @return     JsonResponse       The json response.
     */
    public function destroy(Stock $stock): JsonResponse
    {
        $stock->delete();
        return response()->json([
            'message' => 'Data removed successfully',
        ]);
    }

    /**
     * Stock parameters
     *
     * @return     JsonResponse  The json response.
     */
    public function stockList(): JsonResponse
    {
        return response()->json(
            [
                'products' => ProductSelectResource::collection(Product::get()),
                'warehouses' => WarehouseResource::collection(Warehouse::get()),
            ]
        );
    }
}
