<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Controllers\Api\getInvoiceTerms;
use App\Http\Requests\AppConfigsRequest;
use App\Http\Requests\AppPrinterSetupRequest;
use App\Http\Requests\AppSetupRequest;
use App\Http\Requests\InvoiceTermsRequest;
use App\Http\Requests\NotificationTemplatesRequest;
use App\Http\Resources\Language\LanguageResource;
use App\Http\Resources\UserRoleResource;
use App\Models\Language;
use App\Models\UserRole;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class SettingController extends ApiController
{

    protected $settings;

    protected $collection;

    /**
     * Construct middleware and initialize master app settings
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware('demo')->only(
            [
                'setAppSetup', 'setAppConfigs', 'setAppPrinter',
            ]
        );
        $this->settings = $this->master();
        $this->collection = collect($this->settings);
    }

    /**
     * Gets the application configs.
     *
     * @return     JsonResponse  The application configs.
     */
    public function getAppSetup(): JsonResponse
    {
        return response()->json(
            [
                'logo_url' => $this->settings->getLogoUrl(),
                'background_url' => $this->settings->getBackgroundUrl(),
                'formats' => config('time-date-formats.formats'),
                'timezones' => timezone_identifiers_list(),
                'configs' => $this->collection->only([
                    'app_url',
                    'app_name',
                    'app_https',
                    'app_address',
                    'app_phone',
                    'app_user_registration', 'app_default_role',
                    'app_date_format',
                    'app_timezone', 'app_icon', 'app_background', 'app_timezone',
                    'app_locale',
                    'app_date_locale',
                    'app_date_format',
                    'recaptcha_enabled', 'recaptcha_public', 'recaptcha_private',
                ]),
            ]
        );
    }

    /**
     * Sets the application setup.
     *
     * @param      \App\Http\Requests\AppSetupRequest  $general  The general
     *
     * @return     JsonResponse                                 The json response.
     */
    public function setAppSetup(AppSetupRequest $general): JsonResponse
    {
        $this->settings->update($general->validated());
        return response()->json(
            ['message' => __('Settings updated successfully')]
        );
    }

    /**
     * Gets the configs.
     *
     * @return     JsonResponse  The configs.
     */
    public function getAppConfigs(): JsonResponse
    {

        return response()->json(
            ['data' => $this->collection->only([
                'pos_monies',
                'pos_scanner_focus_autofocus',
                'pos_scanner_focus_duration',
                'pos_scanner_processing_delay',
                'default_warehouse_id',
                'pos_categories_default_numbers',
                'pos_products_default_numbers',
                'mail_from_address',
                'mail_from_name',
                'mail_mailer',
                'mail_encryption',
                'mail_host',
                'mail_password',
                'mail_port',
                'mail_username',
                'mailgun_domain',
                'mailgun_secret',
                'mailgun_endpoint',
                'currency_symbol', 'currency_symbol_on_left', 'currency_shorthand_notation', 'currency_space_between',
                'tax_rate', 'is_tax_fix', 'is_tax_included', 'tax_id',
                'is_vat',
            ]
            ),
                'command_1' => '/usr/local/bin/php  ' . base_path() . '/artisan schedule:run >> /dev/null',
                'command_2' => '/usr/bin/php  ' . base_path() . '/artisan schedule:run >> /dev/null',
            ]
        );
    }

    /**
     * Sets the configs.
     *
     * @param      \App\Http\Requests\AppConfigsRequest  $outgoingMail  The outgoing mail
     *
     * @return     JsonResponse                                    The json response.
     */
    public function setAppConfigs(AppConfigsRequest $outgoingMail): JsonResponse
    {
        $this->settings->update($outgoingMail->validated());
        return response()->json(
            ['message' => __('Settings updated successfully')]
        );
    }

    /**
     * Gets the application printer.
     *
     * @return     JsonResponse  The application printer.
     */
    public function getAppPrinter(): JsonResponse
    {
        return response()->json(
            $this->collection->only(
                [
                    'invoice_setup',
                    'label_setup', 'device_label_setup',
                ]
            )
        );
    }

    /**
     * Sets the application printer.
     *
     * @param      \App\Http\Requests\AppPrinterSetupRequest  $request    The tax
     *
     * @return     JsonResponse                               The json response.
     */
    public function setAppPrinter(AppPrinterSetupRequest $request): JsonResponse
    {
        $this->settings->update($request->validated());
        return response()->json(
            ['message' => __('Settings updated successfully')]
        );
    }

    /**
     * Gets the invoice terms.
     *
     * @return     <type>  The invoice terms.
     */
    public function getInvoiceTerms()
    {
        return response()->json(
            $this->collection->only(
                [
                    'repair_invoice_terms',
                    'bill_invoice_terms',
                    'sale_invoice_terms',
                    'custom_buy_invoice_terms',
                ]
            )
        );
    }

    /**
     * Sets the invoice terms.
     *
     * @param      \App\Http\Requests\InvoiceTermsRequest  $request  The request
     *
     * @return     JsonResponse                            The json response.
     */
    public function setInvoiceTerms(InvoiceTermsRequest $request): JsonResponse
    {
        $this->settings->update($request->validated());
        return response()->json(
            ['message' => __('Settings updated successfully')]
        );
    }

    /**
     * Gets the notification templates.
     *
     * @return     <type>  The notification templates.
     */
    public function getNotificationTemplates()
    {
        return response()->json(
            $this->collection->only(
                [
                    'template_repair_created', 'template_repair_updated',
                    'template_sale_created',
                    'template_refund_created',
                    'template_custom_bill_created',
                    'template_custom_purchase_created',
                ]
            )
        );
    }

    /**
     * Sets the notification templates.
     *
     * @param      \App\Http\Requests\NotificationTemplatesRequest  $request  The request
     *
     * @return     JsonResponse                                     The json response.
     */
    public function setNotificationTemplates(NotificationTemplatesRequest $request): JsonResponse
    {
        $this->settings->update($request->validated());
        return response()->json(
            ['message' => __('Settings updated successfully')]
        );
    }

    /**
     * User roles list
     *
     * @return     JsonResponse  The json response.
     */
    public function userRoles(): JsonResponse
    {
        return response()->json(UserRoleResource::collection(UserRole::all()));
    }

    /**
     * Languages list
     *
     * @return     JsonResponse  The json response.
     */
    public function languages(): JsonResponse
    {
        return response()->json(LanguageResource::collection(Language::all()));
    }

    /**
     * System optimizer
     *
     * @param      Request       $request  The request
     *
     * @return     JsonResponse  The json response.
     */
    public function optimize(Request $request): JsonResponse
    {
        switch ($request->action) {
            case 'optimize':
                \Artisan::call('optimize:clear');
                break;
            case 'cache':
                \Artisan::call('config:cache');
                \Artisan::call('config:clear');
                break;
            case 'storage_link':
                if (File::exists(public_path('/storage'))) {
                    File::delete(public_path('/storage'));
                }
                \Artisan::call('storage:link');
                break;
            case 'update':
                \Artisan::call('install:update');
                break;
            default:
                \Artisan::call('view:clear');
                break;
        }
        return response()->json([
            'message' => __('System performed successfully'),
            'output' => \Artisan::output(),
        ]);
    }
}
