<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\SaleOrderStoreRequest;
use App\Http\Resources\SaleOrderDetailResource;
use App\Models\SaleOrder;
use App\Models\SaleOrderProduct;
use App\Models\Stock;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class SaleOrderController extends ApiController
{
    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $sales = SaleOrder::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => SaleOrderDetailResource::collection($sales->items()),
                'pagination' => $this->pagination($sales),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\SaleOrderStoreRequest  $request  The request
     *
     * @return     JsonResponse                              The json response.
     */
    public function store(SaleOrderStoreRequest $request): JsonResponse
    {
        $validated = $request->validated();

        $validated['tracking'] = time();
        $validated['user_id'] = auth()->user()->id;
        $validated['uuid'] = \Str::orderedUuid();

        try {
            \DB::beginTransaction();
            $sale = SaleOrder::create($validated);
            foreach ($request->items as $item) {
                SaleOrderProduct::create([
                    'uuid' => $sale->uuid,
                    'tracking' => $sale->tracking,
                    'sale_order_id' => $sale->id,
                    'product_id' => $item['id'],
                    'attribute_id' => $item['attribute_id'],
                    'image' => $item['image'],
                    'name' => $item['name'],
                    'title' => $item['title'],
                    'serial_number' => $item['serial_number'],
                    'price' => $item['price'],
                    'cost' => $item['cost'],
                    'qty' => $item['qty'],
                    'sub_total' => $item['sub_total'],
                    'identity' => $item['identity'],
                    'data' => $item['data'],
                    'user_id' => $sale->user_id,
                ]);

                $stock = Stock::where('quantity', '>=', $item['qty'])->where('product_id', $item['id'])->where('product_attribute_id', $item['attribute_id'])->first();

                if (!$stock) {
                    \DB::rollback();
                    return response()->json(
                        [
                            'message' => $item['name'] . ' ' . __('is out of stock or try with less quantity'),
                        ],
                        500
                    );
                }

                $sold_qty = $stock->quantity - $item['qty'];

                $stock->update([
                    'quantity' => $sold_qty,
                    'sold' => $item['qty'] + $stock->sold,
                ]);
            }

            \DB::commit();
            return response()->json([
                'message' => __('Data saved successfully'),
                'sale' => $sale->uuid,
            ]);
        } catch (\Exception $e) {
            \DB::rollback();
            return response()->json(
                [
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\SaleOrder  $saleOrder  The sale order
     *
     * @return     JsonResponse           The json response.
     */
    public function show(SaleOrder $saleOrder): JsonResponse
    {
        return response()->json(new SaleOrderDetailResource($saleOrder));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\SaleOrder  $saleOrder
     * @return \Illuminate\Http\Response
     */
    public function destroy(SaleOrder $saleOrder): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_sales')) {
            return response()->json([
                'message' => __('This action is unauthorized'),
            ], 403);
        }
        $saleOrder->saleOrderProducts()->delete();
        $saleOrder->delete();
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_sales')) {
            return response()->json([
                'message' => __('This action is unauthorized'),
            ], 403);
        }
        $saleOrders = SaleOrder::whereIn('tracking', $request->rows)->get();
        foreach ($saleOrders as $key => $saleOrder) {
            $saleOrder->saleOrderProducts()->delete();
            $saleOrder->delete();
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }
}
