<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\RepairRequest;
use App\Http\Resources\RepairDetailResource;
use App\Http\Resources\RepairTrackDetailResource;
use App\Models\Repair;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class RepairController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin'])->except('track');
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $users = Repair::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));

        return response()->json(
            [
                'items' => RepairDetailResource::collection($users->items()),
                'pagination' => $this->pagination($users),
            ]
        );
    }

    /**
     *
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\RepairRequest  $request  The request
     *
     * @return     JsonResponse                      The json response.
     */
    public function store(RepairRequest $request): JsonResponse
    {
        $validated = $request->validated();
        if ($request->pre_paid > $request->charges) {
            return response()->json(['message' => __('Prepaid amount must be less then or equal to total repair charges')], 422);
        }
        if ($request->cost > $request->charges) {
            return response()->json(['message' => __('Cost amount must be less then to total repair charges')], 422);
        }
        $validated['tracking'] = time();
        $validated['user_id'] = auth()->user()->id;
        $validated['uuid'] = \Str::orderedUuid();

        try {
            \DB::beginTransaction();
            $repair = Repair::create($validated);
            \DB::commit();
            return response()->json([
                'message' => __('Data saved successfully'),
                'repair' => $repair->uuid,
            ]);
        } catch (\Exception $e) {
            \DB::rollback();
            return response()->json(
                [
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\Repair  $repair  The repair
     *
     * @return     JsonResponse        The json response.
     */
    public function show(Repair $repair): JsonResponse
    {
        return response()->json(new RepairDetailResource($repair));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\RepairRequest  $request  The request
     * @param      \App\Models\Repair                $repair   The repair
     *
     * @return     JsonResponse                      The json response.
     */
    public function update(RepairRequest $request, Repair $repair): JsonResponse
    {
        if ($request->cost > $repair->charges) {
            return response()->json(['message' => __('Cost amount must be less then to total repair charges')], 422);
        }
        if ($request->pre_paid > $repair->charges) {
            return response()->json(['message' => __('Prepaid amount must be less then or equal to total repair charges')], 422);
        }
        if ($request->pre_paid < $repair->pre_paid) {
            return response()->json([
                'message' => $repair->pre_paid . ' ' . __('Already paid') . ', ' . __('Please enter') . ' ' . $repair->pre_paid . ' + ' . __('additional amount to be paid'),
            ], 422);
        }

        try {
            \DB::beginTransaction();
            $recovered = $request->pre_paid - $repair->pre_paid;
            $due = $repair->due;
            $repair->update($request->validated());
            \DB::commit();
            return response()->json([
                'message' => __('Data updated successfully'),
            ]);
        } catch (\Exception $e) {
            \DB::rollback();
            return response()->json(
                [
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Destroys the given repair.
     *
     * @param      \App\Models\Repair  $repair  The repair
     *
     * @return     JsonResponse        The json response.
     */
    public function destroy(Repair $repair): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_repairs')) {
            return response()->json([
                'message' => __('You have not permit to perform this request'),
            ], 403);
        }

        $repair->delete();
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_repairs')) {
            return response()->json([
                'message' => __('This action is unauthorized'),
            ], 403);
        }
        $repairOrders = Repair::whereIn('tracking', $request->rows)->get();
        foreach ($repairOrders as $key => $repair) {
            $repair->delete();
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    /**
     * Track repair order
     *
     * @param Request $request request
     *
     * @return JsonResponse
     */
    public function track(Request $request): JsonResponse
    {
        $order = Repair::where('tracking', $request->track_id)->first();
        if (!$order) {
            return response()->json(['message' => __('Track identity code did not match or order not found')]);
        }
        return response()->json(
            [
                'repair' => new RepairTrackDetailResource($order),
            ]
        );
    }
}
