<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\ProductImageRequest;
use App\Http\Requests\ProductStoreRequest;
use App\Http\Requests\ProductUpdateRequest;
use App\Http\Resources\ProductDetailResource;
use App\Http\Resources\ProductLabelResource;
use App\Http\Resources\ProductResource;
use App\Models\CustomProduct;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\Stock;
use DB;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return     JsonResponse  The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $products = Product::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => ProductResource::collection($products->items()),
                'pagination' => $this->pagination($products),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\ProductStoreRequest  $request  The request
     *
     * @return     JsonResponse                            The json response.
     */
    public function store(ProductStoreRequest $request): JsonResponse
    {

        try {

            DB::beginTransaction();
            $product = Product::create($this->productValidated($request));

            if ($request->saleable) {
                CustomProduct::where('uuid', $request->saleable)->first()->update(['is_sold' => true]);
            }
            if ($request->has_variants) {
                foreach ($request->variants as $item) {
                    $title = $item['is_serial_based'] ? $item['title'] : $product->name;
                    $serailNumber = $item['is_serial_based'] ? $item['serial_number'] : null;
                    $identity = $item['is_serial_based'] ? $item['identity'] : null;
                    $quantity = $item['is_serial_based'] ? 1 : $item['quantity'];
                    $warehouse = $item['warehouse_id'] ?? $this->master()->default_warehouse_id;
                    $attribute = ProductAttribute::create([
                        'title' => $item['title'],
                        'sku' => $item['sku'] ?? \Str::sku($title),
                        'upc' => $item['upc'] ?? $this->getUniqueBarcode(),
                        'serial_number' => $serailNumber,
                        'identity' => $identity,
                        'cost' => $item['cost'],
                        'price' => $item['price'],
                        'is_serial_based' => $item['is_serial_based'],
                        'ws_price' => $product->price,
                        'product_id' => $product->id,
                        'supplier_id' => $item['supplier_id'],
                        'warehouse_id' => $warehouse,

                    ]);
                    Stock::create([
                        'tracking' => time(),
                        'uuid' => \Str::orderedUuid(),
                        'product_id' => $product->id,
                        'product_attribute_id' => $attribute->id,
                        'quantity' => $quantity,
                        'warehouse_id' => $warehouse,
                    ]);
                }
            } else {
                Stock::create([
                    'tracking' => time(),
                    'uuid' => \Str::orderedUuid(),
                    'product_id' => $product->id,
                    'quantity' => $product->is_serial_based ? 1 : $request->quantity,
                    'warehouse_id' => $request->warehouse_id ?? $this->master()->default_warehouse_id,
                ]);
            }
            DB::commit();
            return response()->json([
                'message' => __('Data saved successfully'),
                'product' => new ProductDetailResource($product),
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => 'An error occurred while saving data.',
            ], 500);
        }
    }

/**
 *  Display the specified resource.
 *
 * @param      \App\Models\Product  $product  The product
 *
 * @return     JsonResponse         The json response.
 */
    public function show(Product $product): JsonResponse
    {
        return response()->json(new ProductDetailResource($product));
    }

    /**
     * Update the specified resource in storage
     *
     * @param      \App\Http\Requests\ProductUpdateRequest  $request  The request
     * @param      \App\Models\Product                      $product  The product
     *
     * @return     JsonResponse                             The json response.
     */
    public function update(ProductUpdateRequest $request, Product $product): JsonResponse
    {
        DB::beginTransaction();
        try {
            $validated = $request->validated();
            $product->update($validated);
            if ($request->has_variants) {
                foreach ($validated['variants'] as $key => $item) {
                    $title = $item['is_serial_based'] ? $item['title'] : $product->name;
                    $serailNumber = $item['is_serial_based'] ? $item['serial_number'] : null;
                    $identity = $item['is_serial_based'] ? $item['identity'] : null;
                    $warehouse = $item['warehouse_id'] ?? $this->master()->default_warehouse_id;

                    if ($item['id']) {
                        // Updating existing variant
                        $variant = ProductAttribute::where('id', $item['id'])
                            ->where('product_id', $product->id)
                            ->update([
                                'title' => $item['title'],
                                'sku' => $item['sku'] ?? \Str::sku($title),
                                'upc' => $item['upc'] ?? $this->getUniqueBarcode(),
                                'serial_number' => $serailNumber,
                                'identity' => $identity,
                                'cost' => $item['cost'],
                                'price' => $item['price'],
                                'is_serial_based' => $item['is_serial_based'],
                                'ws_price' => $product->price,
                                'product_id' => $product->id,
                                'supplier_id' => $item['supplier_id'],
                                'warehouse_id' => $warehouse,
                            ]);
                    } else {
                        //create new variant
                        $quantity = $item['is_serial_based'] ? 1 : $item['quantity'];
                        $attribute = ProductAttribute::create([
                            'title' => $item['title'],
                            'sku' => $item['sku'] ?? \Str::sku($title),
                            'upc' => $item['upc'] ?? $this->getUniqueBarcode(),
                            'serial_number' => $serailNumber,
                            'identity' => $identity,
                            'cost' => $item['cost'],
                            'price' => $item['price'],
                            'is_serial_based' => $item['is_serial_based'],
                            'ws_price' => $product->price,
                            'product_id' => $product->id,
                            'supplier_id' => $item['supplier_id'],
                            'warehouse_id' => $warehouse,
                        ]);

                        Stock::create([
                            'tracking' => time(),
                            'uuid' => \Str::orderedUuid(),
                            'product_id' => $product->id,
                            'product_attribute_id' => $attribute->id,
                            'quantity' => $quantity,
                            'warehouse_id' => $warehouse,
                        ]);
                    }
                }
            }
            DB::commit();
            return response()->json([
                'message' => __('Data updated successfully'),
                'product' => new ProductDetailResource($product),
            ]);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                'error' => 'An error occurred while saving data.',
            ], 500);
        }
    }

/**
 * Destroys the given product.
 *
 * @param      \App\Models\Product  $product  The product
 *
 * @return     JsonResponse         The json response.
 */
    public function destroy(Product $product): JsonResponse
    {
        $product->productAttributes()->delete();
        foreach ($product->productAttributes as $attribute) {
            $attribute->stocks()->delete();
        }
        $product->delete();
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }

    public function imageUpdate(ProductImageRequest $request, Product $product): JsonResponse
    {
        $product->update(['image' => $request->file('product_image')->store('products', 'public')]);
        return response()->json(
            ['message' => __('Image updated successfully')]
        );
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        $products = Product::whereIn('id', $request->rows)->get();
        foreach ($products as $product) {
            $product->productAttributes()->delete();
            foreach ($product->productAttributes as $attribute) {
                $attribute->stocks()->delete();
            }
            $product->delete();
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    public function getLabelList(Request $request): JsonResponse
    {
        $standardProducts = Product::where('has_variants', false)->filter($request->all())->get();
        $variantProducts = ProductAttribute::filter($request->all())->get();
        $allProducts = $standardProducts->concat($variantProducts);

        return response()->json(ProductLabelResource::collection($allProducts));
    }

    /**
     * Product request validation
     *
     * @param      Object  $request  The request
     *
     * @return     Array   Validated request
     */
    protected function productValidated($request): array
    {

        $validated = $request->validated();
        $validated['slug'] = Str::slug($validated['name']);
        $validated['sku'] = \Str::sku($request->name);
        $validated['upc'] = $request->upc ?? $this->getUniqueBarcode();
        return $validated;
    }
}
