<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\BatchVarientsRequest;
use App\Http\Requests\ProductAttributeStoreRequest;
use App\Http\Requests\ProductAttributeUpdateRequest;
use App\Http\Resources\ProductAttributeResource;
use App\Models\Product;
use App\Models\ProductAttribute;
use App\Models\Stock;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProductAttributeController extends ApiController
{
    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum']);
        $this->middleware('demo')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        //
    }

    /**
     * Store a newly created resource in storage
     *
     * @param      \App\Http\Requests\ProductAttributeStoreRequest  $request  The request
     *
     * @return     JsonResponse                                     The json response.
     */
    public function store(ProductAttributeStoreRequest $request): JsonResponse
    {
        $productAttribute = ProductAttribute::create($this->attributeValidated($request));
        if (!$productAttribute->is_serial_based) {
            return response()->json(
                [
                    'message' => __('Data saved successfully'),
                    'attribute' => new ProductAttributeResource($productAttribute),
                ]
            );
        }
        Stock::create([
            'tracking' => time(),
            'uuid' => Str::orderedUuid(),
            'product_id' => $productAttribute->product_id,
            'product_attribute_id' => $productAttribute->id,
            'quantity' => 1,
            'warehouse_id' => $this->master()->default_warehouse_id,
        ]);

        return response()->json(
            [
                'message' => __('Data saved successfully'),
                'attribute' => new ProductAttributeResource($productAttribute),
            ]
        );
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\ProductAttribute  $productAttribute
     * @return \Illuminate\Http\Response
     */
    public function show(ProductAttribute $productAttribute)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\ProductAttributeUpdateRequest  $request           The request
     * @param      \App\Models\ProductAttribute                      $productAttribute  The product attribute
     *
     * @return     JsonResponse                                      The json response.
     */
    public function update(ProductAttributeUpdateRequest $request, ProductAttribute $productAttribute): JsonResponse
    {
        $productAttribute->update($this->attributeValidated($request));
        return response()->json(
            [
                'message' => __('Data updated successfully'),
            ]
        );
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\ProductAttribute  $productAttribute
     * @return \Illuminate\Http\Response
     */
    public function destroy(ProductAttribute $productAttribute): JsonResponse
    {
        $productAttribute->stocks()->delete();
        $productAttribute->delete();
        return response()->json(
            [
                'message' => __('Data removed successfully'),
            ]
        );
    }

    /**
     * Attribute validation
     *
     * @param      <type>  $request  The request
     *
     * @return     array   ( description_of_the_return_value )
     */
    protected function attributeValidated($request, $isUpdate = false): array
    {
        $validated = $request->validated();
        $validated['sku'] = $request->sku ?? \Str::sku($request->title);
        $validated['upc'] = $request->upc ?? $this->getUniqueBarcode();
        return $validated;
    }

    public function stockAlerts(): JsonResponse
    {
        $products = [];
        foreach (Product::where('has_variants', false)->where('is_serial_based', false)->get() as $product) {
            if ($product->getTotalInStock() <= $product->low_qty_alert) {
                array_push($products, [
                    'id' => $product->id,
                    'name' => $product->name,
                    'sold' => $product->getTotalSoldStock(),
                    'qty' => $product->getTotalInStock(),
                    'alert_at' => $product->low_qty_alert,
                ]);
            }
        }
        $productAttributes = [];
        foreach (ProductAttribute::where('is_serial_based', false)->get() as $attribute) {
            if ($attribute->product && $attribute->getTotalInStock() <= $attribute->product->low_qty_alert) {
                array_push($productAttributes, [
                    'id' => $attribute->id,
                    'name' => $attribute->product->name . ' ' . $attribute->title,
                    'sold' => $attribute->getTotalSoldStock(),
                    'qty' => $attribute->getTotalInStock(),
                    'alert_at' => $attribute->product->low_qty_alert,
                    'is_serial_based' => false,
                ]);
            }
        }
        $collection = collect([$products, $productAttributes]);
        $collapsed = $collection->collapse();
        return response()->json($collapsed->all());
    }

    public function batchStore(BatchVarientsRequest $request, Product $product): JsonResponse
    {
        foreach ($request->variants as $item) {
            $attribute = ProductAttribute::create([
                'title' => $product->name . '-' . $item['identity'],
                'sku' => $item['sku'] ?? Str::sku($product->name),
                'upc' => $item['upc'] ?? $this->getUniqueBarcode(),
                'serial_number' => $item['serial_number'],
                'identity' => $item['identity'],
                'cost' => $request->cost ?? $product->cost,
                'price' => $request->price ?? $product->price,
                'is_serial_based' => true,
                'ws_price' => $product->price,
                'product_id' => $product->id,
                'supplier_id' => $request->supplier_id,
            ]);
            Stock::create([
                'tracking' => time(),
                'uuid' => \Str::orderedUuid(),
                'product_id' => $product->id,
                'product_attribute_id' => $attribute->id,
                'quantity' => 1,
                'warehouse_id' => $request->warehouse_id ?? $this->master()->default_warehouse_id,
            ]);
        }
        return response()->json([
            'message' => __('Data saved successfully'),
        ]);
    }
}
