<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Resources\CustomBillPrintResource;
use App\Http\Resources\CustomProductPrintResource;
use App\Http\Resources\RepairOrderPrintResource;
use App\Http\Resources\ReportBillInvoicesResource;
use App\Http\Resources\ReportCustomProductResource;
use App\Http\Resources\ReportRepairedItemsResource;
use App\Http\Resources\ReportSoldProductsResource;
use App\Http\Resources\SaleOrderPrintResource;
use App\Models\CustomBill;
use App\Models\CustomProduct;
use App\Models\Repair;
use App\Models\SaleOrder;
use App\Models\SaleOrderProduct;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class InvoicePrintController extends ApiController
{
    /**
     * Constructs a new instance.
     */
    public function __construct()
    {
        $this->middleware('auth:sanctum')->only('reportCard');
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\SaleOrder  $saleOrder  The sale order
     *
     * @return     JsonResponse           The json response.
     */
    public function saleOrder(SaleOrder $saleOrder): JsonResponse
    {
        return response()->json(new SaleOrderPrintResource($saleOrder));
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\Repair  $repair  The repair
     *
     * @return     JsonResponse        The json response.
     */
    public function repairOrder(Repair $repair): JsonResponse
    {
        return response()->json(new RepairOrderPrintResource($repair));
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\CustomBill  $customBill  The custom bill
     *
     * @return     JsonResponse            The json response.
     */
    public function customBillOrder(CustomBill $customBill): JsonResponse
    {
        return response()->json(new CustomBillPrintResource($customBill));
    }

    /**
     * Display the specified resource.
     *
     * @param      CustomProduct  $customProduct  The custom product
     *
     * @return     JsonResponse   The json response.
     */
    public function customBuyProduct(CustomProduct $customProduct): JsonResponse
    {
        return response()->json(new CustomProductPrintResource($customProduct));
    }

    public function reportCard(Request $request): JsonResponse
    {
        $dateStart = $request->starting_date;
        $dateEnd = $request->ending_date;
        $user = $request->user_id ?? null;

        if (!$user) {
            $sales = SaleOrder::whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $repairs = Repair::whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $bills = CustomBill::whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $purchases = CustomProduct::whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $sold_items = SaleOrderProduct::whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
        } else {
            $sales = SaleOrder::where('user_id', $user)->whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $repairs = Repair::where('user_id', $user)->whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $bills = CustomBill::where('user_id', $user)->whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $purchases = CustomProduct::where('user_id', $user)->whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
            $sold_items = SaleOrderProduct::where('user_id', $user)->whereBetween('created_at', [$dateStart . " 00:00:00", $dateEnd . " 23:59:59"])->get();
        }

        $foundUser = User::where('id', $user)->first();
        $userName = $foundUser ? $foundUser->name : __('All users');

        return response()->json([
            'message' => __('Report generated successfully'),
            'report' => [
                'total_sales' => $sales->count(),
                'total_repairs' => $repairs->count(),
                'total_custom_bills' => $bills->count(),
                'total_custom_products' => $purchases->count(),
                'total_repairs_amount' => collect($repairs)->sum('total_amount'),
                'total_sales_amount' => collect($sales)->sum('payable_after_all'),
                'total_custom_products_amount' => collect($purchases)->sum('cost'),
                'total_custom_bills_amount' => collect($bills)->sum('charges'),
                'sold_items' => ReportSoldProductsResource::collection($sold_items),
                'repaired_items' => ReportRepairedItemsResource::collection($repairs),
                'custom_bills' => ReportBillInvoicesResource::collection($bills),
                'custom_products' => ReportCustomProductResource::collection($purchases),
                'user' => $userName,
            ],
        ]);
    }
}
