<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\CustomProductRequest;
use App\Http\Resources\CustomProductDetailResource;
use App\Models\CustomProduct;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CustomProductController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware('demo')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $products = CustomProduct::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));

        return response()->json(
            [
                'items' => CustomProductDetailResource::collection($products->items()),
                'pagination' => $this->pagination($products),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\CustomProductRequest  $request  The request
     *
     * @return     JsonResponse                             The json response.
     */
    public function store(CustomProductRequest $request): JsonResponse
    {
        $validated = $this->productRequest($request);
        $validated['key'] = time();
        $validated['user_id'] = auth()->user()->id;
        $validated['uuid'] = \Str::orderedUuid();

        try {
            \DB::beginTransaction();
            $product = CustomProduct::create($validated);
            \DB::commit();
            return response()->json([
                'message' => __('Data saved successfully'),
                'product' => $product->uuid,
            ]);
        } catch (\Exception $e) {
            \DB::rollback();
            return response()->json(
                [
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\CustomProduct  $customProduct  The custom product
     *
     * @return     JsonResponse               The json response.
     */
    public function show(CustomProduct $customProduct): JsonResponse
    {
        return response()->json(new CustomProductDetailResource($customProduct));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\CustomProductRequest  $request        The request
     * @param      \App\Models\CustomProduct                $customProduct  The custom product
     *
     * @return     JsonResponse                             The json response.
     */
    public function update(CustomProductRequest $request, CustomProduct $customProduct): JsonResponse
    {
        $customProduct->update($this->productRequest($request));
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Destroys the given custom product.
     *
     * @param      \App\Models\CustomProduct  $customProduct  The custom product
     *
     * @return     JsonResponse               The json response.
     */
    public function destroy(CustomProduct $customProduct): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_custom_products')) {
            return response()->json([
                'message' => __('You have not permit to perform this request'),
            ], 403);
        }
        $customProduct->delete();
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_custom_products')) {
            return response()->json([
                'message' => __('You have not permit to perform this request'),
            ], 403);
        }

        $products = CustomProduct::whereIn('key', $request->rows)->get();
        foreach ($products as $key => $product) {
            $product->delete();
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    protected function productRequest($request, $photo = null, $docs = null)
    {
        $user = auth()->user();
        $biller = $user->name;
        $validated = $request->validated();
        return $validated;
    }
}
