<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\CustomBillRequest;
use App\Http\Resources\CustomBillDetailResource;
use App\Models\CustomBill;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class CustomBillController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $bills = CustomBill::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));

        return response()->json(
            [
                'items' => CustomBillDetailResource::collection($bills->items()),
                'pagination' => $this->pagination($bills),
            ]
        );
    }

    /**
     *
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\CustomBillRequest  $request  The request
     *
     * @return     JsonResponse                      The json response.
     */
    public function store(CustomBillRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $validated['tracking'] = time();
        $validated['uuid'] = \Str::orderedUuid();
        $validated['user_id'] = auth()->user()->id;

        try {
            \DB::beginTransaction();
            $bill = CustomBill::create($validated);
            \DB::commit();
            return response()->json([
                'message' => __('Data saved successfully'),
                'bill' => $bill->uuid,
            ]);
        } catch (\Exception $e) {
            \DB::rollback();
            return response()->json(
                [
                    'message' => $e->getMessage(),
                ],
                500
            );
        }
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\CustomBill  $customBill  The custom bill
     *
     * @return     JsonResponse            The json response.
     */
    public function show(CustomBill $customBill): JsonResponse
    {
        return response()->json(new CustomBillDetailResource($customBill));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\CustomBillRequest  $request     The request
     * @param      \App\Models\CustomBill                $customBill  The custom bill
     *
     * @return     JsonResponse                          The json response.
     */
    public function update(CustomBillRequest $request, CustomBill $customBill): JsonResponse
    {
        $customBill->update($request->validated());
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Destroys the given custom bill.
     *
     * @param      \App\Models\CustomBill  $customBill  The custom bill
     *
     * @return     JsonResponse            The json response.
     */
    public function destroy(CustomBill $customBill): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_custom_bills')) {
            return response()->json([
                'message' => __('This action is unauthorized'),
            ], 403);
        }
        $customBill->delete();
        return response()->json([
            'message' => __('Data removed successfully'),
        ]);
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        if (!\Auth::user()->userRole->checkPermission('remove_custom_bills')) {
            return response()->json([
                'message' => __('This action is unauthorized'),
            ], 403);
        }
        $customBills = CustomBill::whereIn('tracking', $request->rows)->get();
        foreach ($customBills as $key => $customBill) {
            $customBill->delete();
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }
}
