<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\CategoryStoreRequest;
use App\Http\Requests\CategoryUpdateRequest;
use App\Http\Resources\CategoryResource;
use App\Models\Category;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CategoryController extends ApiController
{

    /**
     * Construct middleware and initiated backups list
     */
    public function __construct()
    {
        $this->middleware(['auth:sanctum', 'admin']);
        $this->middleware('demo')->only(['update', 'destroy']);
    }

    /**
     * Display a listing of the resource.
     *
     * @return     JsonResponse  The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $categories = Category::filter($request->all())
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => CategoryResource::collection($categories->items()),
                'pagination' => $this->pagination($categories),
            ]
        );
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\CategoryStoreRequest  $request  The request
     *
     * @return     JsonResponse                             The json response.
     */
    public function store(CategoryStoreRequest $request): JsonResponse
    {
        $category = Category::create($this->categoryValidated($request));
        return response()->json([
            'message' => __('Data saved successfully'),
            'category' => new CategoryResource($category),
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param      \App\Models\Category  $category  The category
     *
     * @return     JsonResponse          The json response.
     */
    public function show(Category $category): JsonResponse
    {
        return response()->json(new CategoryResource($category));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param      \App\Http\Requests\CategoryUpdateRequest  $request   The request
     * @param      \App\Models\Category                      $category  The category
     *
     * @return     JsonResponse                              The json response.
     */
    public function update(CategoryUpdateRequest $request, Category $category): JsonResponse
    {
        $category->update($this->categoryValidated($request));
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     *  Remove the specified resource from storage.
     *
     * @param      \App\Models\Category  $category  The category
     *
     * @return     JsonResponse          The json response.
     */
    public function destroy(Category $category): JsonResponse
    {
        if ($category->products->count() < 1) {
            $category->delete();
            return response()->json(['message' => __('Data removed successfully')]);
        }

        return response()->json([
            'message' => __('This category is being used with one or more products.'),
        ], 422);
    }

    /**
     * Categories list for certain forms
     *
     * @return     JsonResponse  The json response.
     */
    public function categories(): JsonResponse
    {
        return response()->json(CategoryResource::collection(Category::get()));
    }

    public function destroyBatch(Request $request): JsonResponse
    {
        $categories = Category::whereIn('id', $request->rows)->get();
        foreach ($categories as $key => $category) {
            if ($category->products->count() < 1) {
                $category->delete();
            }
        }
        return response()->json(['message' => __('Data removed successfully')]);
    }

    /**
     * Category request validation
     *
     * @param      Object  $request  The request
     *
     * @return     Array   Validated request
     */
    protected function categoryValidated($request): array
    {
        $data = $request->validated();
        $data['slug'] = Str::slug($data['name']);
        return $data;
    }
}
